#----------------------------------------------------------------------------#
# Copyright 2010, Regular Polygon
# Permission to use this software without fee is hereby granted.
# Distribution or publication of this software, or substantial portions of
# the software, is subject to the expressed written consent of the author.
# THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED 
# WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF 
# MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
#----------------------------------------------------------------------------#
# Name        :   Sphere
# Filename    :   rp_sphere_v11.rb
# Version     :   1.1
# SU Version  :   6.0+
# Type        :   Dialog Box
# Description :   Creates a sphere component. 
# Menu Item   :   Draw -> Sphere
# Context Menu:   N/A
# Usage       :   Fill in the dialog box.
# Date        :   7/1/2010
# Author      :   Regular Polygon
# Email       :   regular.polygon@gmail.com
# Website     :   http://regularpolygon.blogspot.com/
#----------------------------------------------------------------------------#

require 'sketchup.rb'

module RP
#----------------------------------------------------------------------------#
class Sphere
	# Default dialog box parameters
	@@radius = (Sketchup.read_default 'RP_sphere', 'radius', '12').to_f.inch
	@@rings = (Sketchup.read_default 'RP_sphere', 'rings', '17').to_i         # number of latitude lines
	@@edges = (Sketchup.read_default 'RP_sphere', 'edges', '24').to_i         # number of longitude lines (edges per latitude line)
	@@flag = Sketchup.read_default 'RP_sphere', 'flag', 'Smooth'              # edge properties
	
	def initialize(radius, rings, edges, flag = 'Smooth')
		@radius, @rings, @edges, @flag = radius, rings, edges, flag
		# make geomentry creation undoable
		Sketchup.active_model.start_operation "Definition"  
			 create_definition
		Sketchup.active_model.commit_operation 
	end

	# This method handles the Draw -> Sphere menu item.
	def self.dialog
		# Show dialog to get user input for Sphere paramters
		prompts = ["球形半径 Radius: ", "纬度值 Latitude Lines: ", "经度值 Longitude Lines: ", "柔化平滑 Edge Properties: "]
		defaults = [@@radius, @@rings, @@edges, @@flag]
		list = [nil, nil, nil, 'Smooth|Soft|Visible']
		results = nil
		# Keep prompting user until input is valid.
		begin
			results = UI.inputbox prompts, defaults, list, "Sphere Parameters"
			return unless results  # user cancelled the operation
			radius, rings, edges = results
			raise "Invalid radius: #{ radius }"  if ( radius <= 0 )
			raise "Enter at least 1 line of latitude."  if ( rings < 1 )
			raise "Enter at least 3 lines of longitude."  if ( edges < 3 )
		rescue
			UI.messagebox $!.message
			retry
		end
		
		# Create the Sphere component and place it in the model.
		self.new(*results).place_component
		# Remember parameters
		@@radius, @@rings, @@edges, @@flag = results
		# Remember parameters between sessions
		Sketchup.write_default 'RP_sphere', 'radius', @@radius.inspect
		Sketchup.write_default 'RP_sphere', 'rings', @@rings.to_s
		Sketchup.write_default 'RP_sphere', 'edges', @@edges.to_s
		Sketchup.write_default 'RP_sphere', 'flag', @@flag
	end

	# Place an instance in the model using the component placement tool.
	def place_component
		Sketchup.active_model.place_component @definition 
	end
	
	private  #-----------------------------------------------------------------#

	# Create a Sphere component definition
	def create_definition
		flag = { 'Visible' => 0, 'Soft' => 4, 'Smooth' => 12 }[ @flag ]
		@definition = Sketchup.active_model.definitions.add "Sphere"
		@definition.entities.fill_from_mesh mesh, false, flag
		@definition.insertion_point = Geom::Point3d.new(0, 0, -@radius)
	end

	def mesh
		# create mesh
		numpts = @edges * @rings + 2       # number of points in the mesh
		numpoly = @edges * (@rings + 1)    # number of polygons in the mesh
		mesh = Geom::PolygonMesh.new(numpts, numpoly) 
		# Add pole points to mesh
		north_pole = mesh.add_point([0, 0, +@radius])
		south_pole = mesh.add_point([0, 0, -@radius])
		# Add the other points to the mesh
		index_array = []
		for i in 1..@rings do
			phi = i * Math::PI / (@rings + 1)   # polar angle
			indices = []
			for j in 0...@edges do
				theta = j * 2 * Math::PI / @edges       # azimuthal angle
				x = @radius * Math.cos(theta) * Math.sin(phi)
				y = @radius * Math.sin(theta) * Math.sin(phi)
				z = @radius * Math.cos(phi)
				indices.push( mesh.add_point([x, y, z]) )
			end
			index_array.push indices
		end
		# Add top row of polygons
		i1 = index_array.first
		for j in 0...@edges do
			k = (j+1) % @edges
			mesh.add_polygon i1[j], i1[k], north_pole
		end
		# Add inbetween rows of polygons
		i1 = index_array[0]
		for i in 1...@rings do
			i2 = index_array[i]
			for j in 0...@edges do
				k = (j+1) % @edges
				# Since latatitude lines are parallel, these 4 points are co-planar.
				mesh.add_polygon i2[j], i2[k], i1[k], i1[j]
			end
			i1 = i2
		end
		# Add bottom row of polygons
		i1 = index_array.last
		for j in 0...@edges do
			k = (j+1) % @edges
			mesh.add_polygon south_pole, i1[k], i1[j]
		end
		return mesh
	end
	
end  # class Sphere
end  # module RP

# Menu
#----------------------------------------------------------------------------#
unless file_loaded? 'rp_sphere'
	add_separator_to_menu("Draw")
	UI.menu("Draw").add_item("参数球形 Sphere") { RP::Sphere.dialog } 
end
file_loaded 'rp_sphere'
